﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace DocumentProcessing
{
    class DocumentProcessor
    {
        // Listing 5-29 (część 1). Zdarzenie, które można odwołać
        public event EventHandler<ProcessCancelEventArgs> Processing;
        public event EventHandler<ProcessEventArgs> Processed;

        public Func<Document, string> LogTextProvider
        {
            get;
            set;
        }


        class ActionCheckPair
        {
            public Action<Document> Action { get; set; }
            public Predicate<Document> QuickCheck { get; set; }
        }

        private readonly List<ActionCheckPair> processes =
            new List<ActionCheckPair>();

        public void AddProcess(Action<Document> action)
        {
            AddProcess(action, null);
        }

        public void AddProcess(Action<Document> action,
            Predicate<Document> quickCheck)
        {
            processes.Add(
            new ActionCheckPair { Action = action, QuickCheck = quickCheck });
        }

        // Listing 5-30. Obsługa odwołania operacji
        public void Process(Document doc)
        {
            ProcessEventArgs e = new ProcessEventArgs(doc);
            ProcessCancelEventArgs ce = new ProcessCancelEventArgs(doc);
            OnProcessing(ce);
            if (ce.Cancel)
            {
                Console.WriteLine("Proces został anulowany.");
                if (LogTextProvider != null)
                {
                    Console.WriteLine(LogTextProvider(doc));
                }
                return;
            }

            // Najpierw przeprowadzamy szybką weryfikację dokumentu.
            foreach (ActionCheckPair process in processes)
            {
                if (process.QuickCheck != null && !process.QuickCheck(doc))
                {
                    Console.WriteLine("Przetwarzanie nie zakończy się pomyślnie.");
                    if (LogTextProvider != null)
                    {
                        Console.WriteLine(LogTextProvider(doc));
                    }
                    OnProcessed(e);
                    return;
                }
            }
            // Teraz wykonujemy akcję
            foreach (ActionCheckPair process in processes)
            {
                process.Action(doc);
                if (LogTextProvider != null)
                {
                    Console.WriteLine(LogTextProvider(doc));
                }
            }
            OnProcessed(e);
        }

        // Listing 5-29 (część 2). Zdarzenie, które można odwołać
        private void OnProcessing(ProcessCancelEventArgs e)
        {
            if (Processing != null)
            {
                Processing(this, e);
            }
        }

        private void OnProcessed(ProcessEventArgs e)
        {
            if (Processed != null)
            {
                Processed(this, e);
            }
        }
    }
}